from django.contrib import admin
from django.db.models import Sum
from .models import Wallet, Transaction

# 1. Inline View: Show Transactions inside the Wallet Page
class TransactionInline(admin.TabularInline):
    model = Transaction
    extra = 0
    readonly_fields = ('amount', 'transaction_type', 'description', 'created_at')
    can_delete = False
    ordering = ('-created_at',)
    max_num = 10  # Limit to showing the last 10 for performance

    def has_add_permission(self, request, obj=None):
        return False

# 2. Wallet Admin
@admin.register(Wallet)
class WalletAdmin(admin.ModelAdmin):
    list_display = ('get_username', 'balance', 'is_frozen', 'updated_at', 'created_at')
    list_filter = ('is_frozen', 'created_at')
    search_fields = ('user__username', 'user__email')
    readonly_fields = ('balance', 'created_at', 'updated_at')
    inlines = [TransactionInline]
    
    # Custom method to show username clearly
    def get_username(self, obj):
        return obj.user.username
    get_username.short_description = 'User'
    get_username.admin_order_field = 'user__username'

    # Actions to Bulk Freeze/Unfreeze
    actions = ['freeze_wallets', 'unfreeze_wallets']

    @admin.action(description='Freeze selected wallets')
    def freeze_wallets(self, request, queryset):
        queryset.update(is_frozen=True)

    @admin.action(description='Unfreeze selected wallets')
    def unfreeze_wallets(self, request, queryset):
        queryset.update(is_frozen=False)

# 3. Transaction Admin
@admin.register(Transaction)
class TransactionAdmin(admin.ModelAdmin):
    list_display = ('id', 'get_wallet_user', 'amount', 'transaction_type', 'created_at')
    list_filter = ('transaction_type', 'created_at')
    search_fields = ('wallet__user__username', 'wallet__user__email', 'id', 'description')
    readonly_fields = ('wallet', 'amount', 'transaction_type', 'description', 'created_at')
    
    # Prevent deleting transaction history to maintain audit trail
    def has_delete_permission(self, request, obj=None):
        return False

    def get_wallet_user(self, obj):
        return obj.wallet.user.username
    get_wallet_user.short_description = 'User'