from rest_framework.views import APIView
from django.http import HttpResponseRedirect
from rest_framework.permissions import AllowAny
from drf_yasg.utils import swagger_auto_schema
from drf_yasg import openapi
from .services import TrackingService

class RedirectView(APIView):
    """
    The main entry point for Affiliate Traffic.
    1. Extracts Campaign ID and Affiliate ID.
    2. Logs the click.
    3. 302 Redirects user to the Campaign Target URL.
    """
    permission_classes = [AllowAny]

    @swagger_auto_schema(
        operation_description="Intercepts affiliate link, logs click, and redirects to target URL.",
        responses={
            302: openapi.Response(description="Redirects to Campaign Target URL"),
            404: openapi.Response(description="Campaign or Affiliate not found")
        }
    )
    def get(self, request, campaign_id, affiliate_id):
        # 1. Gather Metadata
        meta = {
            'ip_address': self._get_client_ip(request),
            'user_agent': request.META.get('HTTP_USER_AGENT', ''),
            'referer': request.META.get('HTTP_REFERER', '')
        }

        # 2. Log and Get Target (Delegated to Service)
        target_url = TrackingService.log_click(
            campaign_id=campaign_id,
            affiliate_id=affiliate_id,
            meta=meta
        )

        # 3. Perform 302 Redirect
        return HttpResponseRedirect(target_url)

    def _get_client_ip(self, request):
        x_forwarded_for = request.META.get('HTTP_X_FORWARDED_FOR')
        if x_forwarded_for:
            ip = x_forwarded_for.split(',')[0]
        else:
            ip = request.META.get('REMOTE_ADDR')
        return ip