from django.contrib import admin
from django.apps import apps
from django.db import models

# ==============================================================================
#  Universal Auto-Admin Configuration
#  This script finds all models in the 'tracking' app and registers them
#  with a full dashboard (columns, filters, and search).
# ==============================================================================

try:
    # 1. Get the app configuration for 'tracking'
    # 'tracking' is the label usually defined in apps.py. 
    # If your app label is different (e.g., 'apps_tracking'), change it here.
    app_config = apps.get_app_config('tracking')
    
    # 2. Iterate through every model found in this app
    for model in app_config.get_models():
        
        # Check if already registered to avoid conflicts
        if admin.site.is_registered(model):
            continue

        # 3. Dynamically build a customized ModelAdmin class
        # This introspection allows us to build the UI based on your actual data fields
        
        # Get all field names to display in the table columns
        all_fields = [field.name for field in model._meta.fields]
        
        # specific fields for the sidebar filters (Dates, Booleans, ForeignKeys)
        filter_fields = [
            field.name for field in model._meta.fields 
            if field.get_internal_type() in [
                'BooleanField', 
                'DateField', 
                'DateTimeField', 
                'ForeignKey', 
                'ChoiceField'
            ]
        ]
        
        # Specific fields for the search bar (Text and Char fields)
        searchable_fields = [
            field.name for field in model._meta.fields 
            if field.get_internal_type() in ['CharField', 'TextField', 'EmailField']
        ]

        # Define the Admin class dynamically
        class UniversalAdmin(admin.ModelAdmin):
            list_display = all_fields
            list_display_links = all_fields[:1] # Make the first column clickable
            list_filter = filter_fields
            search_fields = searchable_fields
            
            # Optional: Add read-only fields for standard tracking timestamps if they exist
            readonly_fields = [f for f in all_fields if f in ['created_at', 'updated_at', 'timestamp']]

        # 4. Register the model with the dynamic admin class
        admin.site.register(model, UniversalAdmin)

except LookupError:
    print("Error: Could not find app label 'tracking'. Check apps/tracking/apps.py name.")