from aiogram import Router, F, types
from asgiref.sync import sync_to_async

from apps.telegram_bot.keyboards.inline import get_campaigns_keyboard, get_main_menu_keyboard
from apps.telegram_bot.services import format_campaign_message

# Business Logic Imports
from apps.campaigns.services import get_active_campaigns
from apps.tracking.services import generate_deep_link

router = Router()

@router.callback_query(F.data == "cmd_campaigns")
async def list_campaigns(callback: types.CallbackQuery, user):
    """
    Fetch active campaigns from Django DB and display as list.
    """
    if not user:
        await callback.answer("Please register first.", show_alert=True)
        return

    # Call Campaigns App Service
    # Note: list() is needed because QuerySets are lazy/async-incompatible directly
    campaigns = await sync_to_async(lambda: list(get_active_campaigns()))()
    
    if not campaigns:
        await callback.message.edit_text(
            "📭 No active campaigns are available right now. Please check back later.",
            reply_markup=get_main_menu_keyboard(is_affiliate=True)
        )
    else:
        # We might only show the first 5 or use pagination (skipped for brevity)
        await callback.message.edit_text(
            "📢 <b>Active Campaigns</b>\nSelect one to get your tracking link:",
            reply_markup=get_campaigns_keyboard(campaigns),
            parse_mode="HTML"
        )
    
    await callback.answer()

@router.callback_query(F.data.startswith("get_link:"))
async def generate_affiliate_link(callback: types.CallbackQuery, user):
    """
    Calls Tracking App to generate a unique link for this user + campaign.
    """
    campaign_id = callback.data.split(":")[1]

    try:
        # Call Tracking App Service
        # returns string e.g., "https://tabligh.show/t/xyz123"
        link_url = await sync_to_async(generate_deep_link)(
            user_id=user.id,
            campaign_id=campaign_id
        )

        await callback.message.answer(
            f"🔗 <b>Your Tracking Link</b>\n\n"
            f"Copy the link below and share it:\n"
            f"<code>{link_url}</code>",
            parse_mode="HTML"
        )
        await callback.answer("Link generated!", show_alert=False)

    except Exception as e:
        await callback.answer("Error generating link.", show_alert=True)
        # Log the exception strictly