from rest_framework import viewsets, mixins, status
from rest_framework.response import Response
from rest_framework.decorators import action
from rest_framework.permissions import IsAdminUser
from drf_spectacular.utils import extend_schema, OpenApiParameter

from .models import ImportJob
from .serializers import ImportJobSerializer
from .tasks import process_import_task

@extend_schema(tags=['Data Processor'])
class ImportJobViewSet(viewsets.ModelViewSet):
    """
    API for uploading Excel files and tracking their processing status.
    Only Admins should access this.
    """
    queryset = ImportJob.objects.all().order_by('-created_at')
    serializer_class = ImportJobSerializer
    permission_classes = [IsAdminUser]

    def perform_create(self, serializer):
        job = serializer.save()
        # Auto-trigger task on upload
        process_import_task.delay(job.id)

    @extend_schema(
        summary="Retry a failed or pending job",
        request=None,
        responses={200: None}
    )
    @action(detail=True, methods=['post'])
    def retry(self, request, pk=None):
        job = self.get_object()
        if job.status == ImportJob.Status.PROCESSING:
            return Response(
                {"detail": "Job is already processing."}, 
                status=status.HTTP_400_BAD_REQUEST
            )
        
        job.status = ImportJob.Status.PENDING
        job.save()
        process_import_task.delay(job.id)
        return Response({"detail": "Job queued for processing."}, status=status.HTTP_200_OK)