import json
from django.contrib import admin
from django.utils.html import format_html
from django.utils.safestring import mark_safe
from django.utils.translation import gettext_lazy as _

# Import the model and the task
from .models import ImportJob
from .tasks import process_import_task

@admin.action(description='⚡ Run Processing for selected Jobs')
def run_processing_action(modeladmin, request, queryset):
    """
    Admin Action to manually trigger the Celery task for selected jobs.
    """
    count = 0
    for job in queryset:
        # Only trigger if not currently processing to avoid duplicates
        if job.status != ImportJob.Status.PROCESSING:
            # Reset status to PENDING so the UI shows it's queued
            job.status = ImportJob.Status.PENDING
            job.save(update_fields=['status'])
            
            # Trigger Celery Task
            process_import_task.delay(job.id)
            count += 1
            
    if count > 0:
        modeladmin.message_user(request, f"Successfully queued {count} jobs for processing.", level='SUCCESS')
    else:
        modeladmin.message_user(request, "No jobs were queued (selected jobs might already be processing).", level='WARNING')

@admin.register(ImportJob)
class ImportJobAdmin(admin.ModelAdmin):
    """
    Advanced Admin configuration for managing Excel Import Jobs.
    """
    # ----------------------------------------------------------------------
    # LIST VIEW CONFIGURATION
    # ----------------------------------------------------------------------
    list_display = (
        'id', 
        'campaign_link', 
        'file_download_link', 
        'status_badge', 
        'created_at', 
        'log_summary_short'
    )
    list_display_links = ('id', 'campaign_link')
    
    list_filter = (
        'status', 
        'created_at', 
        ('campaign', admin.RelatedOnlyFieldListFilter), # Efficient filter for FK
    )
    
    search_fields = (
        'id', 
        'campaign__title', 
        'processing_log'
    )
    
    actions = [run_processing_action]
    
    date_hierarchy = 'created_at'
    ordering = ('-created_at',)
    list_per_page = 20

    # ----------------------------------------------------------------------
    # DETAIL/EDIT VIEW CONFIGURATION
    # ----------------------------------------------------------------------
    readonly_fields = (
        'status_badge_large', 
        'formatted_processing_log', 
        'created_at', 
        'updated_at'
    )

    fieldsets = (
        ("Job Details", {
            "fields": (
                'campaign', 
                'file', 
                'status', 
                'status_badge_large'
            )
        }),
        ("Results & Logs", {
            "fields": (
                'formatted_processing_log',
            ),
            "classes": ('collapse', 'open'), # Keep open by default
        }),
        ("Timestamps", {
            "fields": ('created_at', 'updated_at'),
            "classes": ('collapse',),
        }),
    )

    # ----------------------------------------------------------------------
    # CUSTOM DISPLAY METHODS
    # ----------------------------------------------------------------------

    @admin.display(description='Campaign')
    def campaign_link(self, obj):
        """Link to the related Campaign edit page."""
        if obj.campaign:
            return obj.campaign.title
        return "-"

    @admin.display(description='Excel File')
    def file_download_link(self, obj):
        """Clickable link to download the uploaded file."""
        if obj.file:
            # Returns a clickable HTML link
            return format_html(
                '<a href="{}" target="_blank" style="font-weight:bold;">⬇ Download</a>',
                obj.file.url
            )
        return "-"

    @admin.display(description='Status')
    def status_badge(self, obj):
        """Color-coded status badge for the list view."""
        colors = {
            ImportJob.Status.PENDING: 'gray',
            ImportJob.Status.PROCESSING: 'orange',
            ImportJob.Status.COMPLETED: 'green',
            ImportJob.Status.FAILED: 'red',
        }
        color = colors.get(obj.status, 'black')
        return format_html(
            '<span style="color: white; background-color: {}; padding: 3px 10px; border-radius: 10px; font-weight: bold; font-size: 11px;">{}</span>',
            color,
            obj.get_status_display().upper()
        )

    @admin.display(description='Current Status')
    def status_badge_large(self, obj):
        """Larger status badge for the detail view."""
        return self.status_badge(obj)

    @admin.display(description='Processing Log (JSON)')
    def formatted_processing_log(self, obj):
        """Pretty-print the JSON log so it's readable in the admin."""
        if not obj.processing_log:
            return "-"
        
        # Convert dict to nicely indented JSON string
        json_str = json.dumps(obj.processing_log, indent=4, sort_keys=True)
        
        # Render inside a <pre> tag for code formatting
        style = "background-color: #f5f5f5; padding: 10px; border-radius: 5px; border: 1px solid #ccc; max-height: 400px; overflow-y: auto;"
        return format_html('<pre style="{}">{}</pre>', style, json_str)

    @admin.display(description='Results')
    def log_summary_short(self, obj):
        """Quick summary for the list view column."""
        if not obj.processing_log:
            return "-"
        
        data = obj.processing_log
        # Safe access to dictionary keys
        total = data.get('total_rows', 0)
        success = data.get('processed', 0)
        errors = len(data.get('errors', []))
        
        if errors > 0:
            return format_html(
                '<span style="color: red;">❌ {} Errors</span> (Ok: {})', 
                errors, success
            )
        if success > 0 and success == total:
             return format_html('<span style="color: green;">✔ All {} OK</span>', success)
             
        return f"Processed: {success}/{total}"