import importlib
import re

def normalize_phone_number(phone: str) -> str:
    """
    Normalizes diverse phone number formats to the standard Iranian 0912... format.
    Example: 
    +98 912 123 4567 -> 09121234567
    00989121234567   -> 09121234567
    9121234567       -> 09121234567
    """
    if not phone:
        return ""
    
    # Remove all non-digit characters
    digits = re.sub(r'\D', '', str(phone))
    
    # Handle country codes (Iran +98)
    if digits.startswith('98'):
        digits = digits[2:]
    elif digits.startswith('0098'):
        digits = digits[4:]
        
    # Ensure it starts with 0 if it's a mobile number (starts with 9)
    if digits.startswith('9'):
        digits = '0' + digits
        
    return digits

def dynamic_import(module_path: str, class_name: str):
    """
    Dynamically imports a class from a module path.
    Usage: dynamic_import('apps.accounts.services', 'OTPService')
    """
    try:
        module = importlib.import_module(module_path)
        return getattr(module, class_name)
    except (ImportError, AttributeError) as e:
        raise ImportError(f"Could not import {class_name} from {module_path}: {e}")