from rest_framework.renderers import JSONRenderer

class TablighShowJSONRenderer(JSONRenderer):
    """
    Custom Renderer to wrap all API responses in a standard envelope:
    {
        "status": "success" | "error",
        "code": 200,
        "message": "OK" | Error Details,
        "data": ...
    }
    """
    def render(self, data, accepted_media_type=None, renderer_context=None):
        status_code = 200
        message = "OK"
        status = "success"

        if renderer_context:
            response = renderer_context.get('response')
            if response:
                status_code = response.status_code

        # Determine status based on code
        if status_code >= 400:
            status = "error"
            message = "An error occurred"
            
            # If data contains a "detail" key (common in DRF exceptions), use it as message
            if isinstance(data, dict) and 'detail' in data:
                message = data.pop('detail')
            
            # If the custom exception handler already formatted the data, use it
            if isinstance(data, dict) and 'message' in data:
                message = data.get('message', message)

        response_data = {
            "status": status,
            "code": status_code,
            "message": message,
            "data": data
        }

        return super().render(response_data, accepted_media_type, renderer_context)