from rest_framework.views import exception_handler
from rest_framework.exceptions import APIException
from rest_framework.response import Response
from rest_framework import status

class ServiceException(Exception):
    """
    Base class for business logic errors in the Service Layer.
    These are caught by the handler and converted to 400 Bad Request.
    """
    def __init__(self, message, code=None):
        self.message = message
        self.code = code
        super().__init__(message)

def custom_exception_handler(exc, context):
    """
    Custom exception handler that standardizes error responses.
    """
    # Call REST framework's default exception handler first
    response = exception_handler(exc, context)

    # Handle custom ServiceExceptions (Business Logic Errors)
    if isinstance(exc, ServiceException):
        response = Response(
            {"detail": exc.message},
            status=status.HTTP_400_BAD_REQUEST
        )

    # Handle standard Python exceptions (prevents 500 HTML leaks)
    if response is None:
        # In production, you might want to log the full stack trace here
        response = Response(
            {"detail": str(exc)},
            status=status.HTTP_500_INTERNAL_SERVER_ERROR
        )

    # The Renderer (TablighShowJSONRenderer) will wrap this Response data
    # into the {"status": "error", ...} envelope.
    
    return response