# backend/apps/campaigns/views.py
from rest_framework import generics, permissions
from rest_framework.response import Response
from rest_framework.views import APIView
from rest_framework import status
from drf_spectacular.utils import extend_schema, OpenApiParameter

from .models import Campaign
from .serializers import CampaignListSerializer, CampaignDetailSerializer
from .services import CampaignService

class CampaignListView(generics.ListAPIView):
    """
    List all active campaigns.
    Supports filtering by ?category=<slug>.
    """
    permission_classes = [permissions.IsAuthenticated] # Or AllowAny depending on business rules
    serializer_class = CampaignListSerializer

    @extend_schema(
        summary="List Active Campaigns",
        description="Retrieve a list of all active campaigns with optional category filtering.",
        parameters=[
            OpenApiParameter(name="category", description="Filter by category slug", required=False, type=str),
        ],
        responses={200: CampaignListSerializer(many=True)}
    )
    def get_queryset(self):
        category_slug = self.request.query_params.get('category')
        return CampaignService.get_active_campaigns(category_slug=category_slug)


class CampaignDetailView(generics.RetrieveAPIView):
    """
    Retrieve details for a specific campaign, including media assets.
    """
    permission_classes = [permissions.IsAuthenticated]
    serializer_class = CampaignDetailSerializer
    lookup_field = 'slug'
    queryset = Campaign.objects.filter(is_active=True)

    @extend_schema(
        summary="Get Campaign Details",
        description="Retrieve full details, including target URL and media assets (banners/videos).",
        responses={200: CampaignDetailSerializer}
    )
    def get(self, request, *args, **kwargs):
        return super().get(request, *args, **kwargs)