import random
from django.utils import timezone
from datetime import timedelta
from rest_framework_simplejwt.tokens import RefreshToken
from django.db import transaction
from .models import CustomUser, OTPCode

class OTPService:
    @staticmethod
    def generate_otp(phone_number: str) -> str:
        """
        Generates a 5-digit code, saves to DB, and simulates sending.
        """
        # Generate random 5 digit code
        code = str(random.randint(10000, 99999))
        expires_at = timezone.now() + timedelta(minutes=5)
        
        OTPCode.objects.create(
            phone_number=phone_number,
            code=code,
            expires_at=expires_at
        )
        
        # SIMULATED SMS
        print(f"======================================")
        print(f" [SMS MOCK] To: {phone_number} | Code: {code}")
        print(f"======================================")
        
        return code

    @staticmethod
    def validate_otp(phone_number: str, code: str) -> bool:
        """
        Validates the latest code for the phone number.
        """
        otp_record = OTPCode.objects.filter(
            phone_number=phone_number,
            code=code,
            is_verified=False,
            expires_at__gt=timezone.now()
        ).first()

        if otp_record:
            otp_record.is_verified = True
            otp_record.save()
            return True
        return False

class AuthService:
    @staticmethod
    def get_tokens_for_user(user):
        refresh = RefreshToken.for_user(user)
        return {
            'refresh': str(refresh),
            'access': str(refresh.access_token),
        }

    @staticmethod
    @transaction.atomic
    def authenticate_or_register(phone_number: str, referral_code: str = None) -> dict:
        """
        Finds user by phone or creates a new one. 
        Handles referral linking if new user.
        """
        user, created = CustomUser.objects.get_or_create(
            phone_number=phone_number
        )

        if created and referral_code:
            # Try to link to an affiliate
            try:
                referrer = CustomUser.objects.get(referral_code=referral_code)
                user.invited_by = referrer
                user.save(update_fields=['invited_by'])
            except CustomUser.DoesNotExist:
                pass # Invalid referral code ignored silently

        tokens = AuthService.get_tokens_for_user(user)
        
        return {
            'user': user,
            'tokens': tokens,
            'is_new_user': created
        }