import uuid
from django.db import models
from django.contrib.auth.models import AbstractBaseUser, PermissionsMixin
from apps.common.models import TimeStampedModel
from .managers import CustomUserManager

class CustomUser(AbstractBaseUser, PermissionsMixin, TimeStampedModel):
    """
    Core User model identified by phone_number.
    Supports referral hierarchy via 'invited_by'.
    """
    phone_number = models.CharField(max_length=20, unique=True, db_index=True)
    is_affiliate = models.BooleanField(default=False, help_text="Is this user an active affiliate?")
    
    # Referral Logic
    referral_code = models.CharField(max_length=50, unique=True, blank=True)
    invited_by = models.ForeignKey(
        'self', 
        on_delete=models.SET_NULL, 
        null=True, 
        blank=True, 
        related_name='referrals',
        help_text="The affiliate who invited this user."
    )

    # Telegram Integration
    telegram_id = models.BigIntegerField(unique=True, null=True, blank=True)

    # Django Admin / Auth fields
    is_staff = models.BooleanField(default=False)
    is_active = models.BooleanField(default=True)

    objects = CustomUserManager()

    USERNAME_FIELD = 'phone_number'
    REQUIRED_FIELDS = []

    def __str__(self):
        return self.phone_number

    def save(self, *args, **kwargs):
        # Auto-generate unique referral code if missing
        if not self.referral_code:
            self.referral_code = str(uuid.uuid4())[:8].upper()
        super().save(*args, **kwargs)


class OTPCode(TimeStampedModel):
    """
    Ephemeral storage for OTP codes.
    """
    phone_number = models.CharField(max_length=20)
    code = models.CharField(max_length=6)
    expires_at = models.DateTimeField()
    is_verified = models.BooleanField(default=False)

    class Meta:
        ordering = ['-created_at']

    def __str__(self):
        return f"{self.phone_number} - {self.code}"